/** @file   game.cpp
 * @brief   Implementation of Game - class.
 * @version $Revision: 1.6 $
 * @author  Tomi Lamminsaari
 */
 
 
#include "game.h"
#include "consts.h"
#include "war.h"
#include "gameanims.h"
#include "settings.h"
#include "weapon.h"
#include "menumain.h"
#include "menusettings.h"
#include "menulevelchange.h"
#include "menusavegame.h"
#include "menuloadgame.h"
#include "menunotification.h"
#include "highscoresmenu.h"
#include "enterhighscoresmenu.h"
#include "filehasher.h"
#include "eng2d.h"
#include "www_assert.h"
#include "warglobals.h"
#include "highscoretable.h"
#include "soundsamples.h"
#include "progressindicator.h"
#include "GfxManager.h"
#include <string>
#include <fstream>
#include <almp3.h>
using namespace eng2d;
using std::ifstream;
using std::ofstream;
using std::endl;
using std::string;

namespace WeWantWar {


const int KPlaybackVolume = 190;

//********************************************************************
//                                                                   *
//      Constructors, destructor and operators                       *
//                                                                   *
//********************************************************************

bool Game::gameHasFocus = true;

Game::Game() :
  m_quitNow( false ),
  m_titlesong( 0 ),
  m_pPage( 0 ),
  m_nextLevel( 0 )
{
}

Game::~Game()
{
}


//********************************************************************
//                                                                   *
//      Public interface                                             *
//                                                                   *
//********************************************************************

int Game::init()
{
  FpsTimer::install( 40 );
  
  LOG_MESSAGE( "Init constants..." );
  Consts::init();
  LOG_MESSAGE( "done" );
  
  LOG_MESSAGE( "Loading configuration..." );
  if ( Settings::init() != 0 ) {
    return -1;
  }
  LOG_MESSAGE( "done" );
  
  LOG_MESSAGE( "Loading graphics..." );
  if ( GfxManager::init() != 0 ) {
    return -1;
  }
  LOG_MESSAGE( "done" );
  
  ProgressIndicator ind(8);
  ind.position(26,180);
  ind.size(380,5);
  ind.setStep(6);
  
  LOG_MESSAGE( "Creating animations..." );
  if ( GameAnims::init() != 0 ) {
    return -1;;
  }
  LOG_MESSAGE( "done" );
  ind.advance();
  ind.redraw();
  
  LOG_MESSAGE( "Loading sounds..." );
  if ( Sound::initSamples( "data/soundsamples.txt" ) != 0 ) {
    return -1;
  }
  LOG_MESSAGE( "done" );
  ind.advance();
  ind.redraw();
  

  Weapon::init();
  
  LOG_MESSAGE( "Creating mainmenu..." );
  m_pPage = new MenuMain();
  LOG_MESSAGE( "done" );
  
  return 0;
}


void Game::run()
{
  LOG_MESSAGE( "Loading title song..." );
  m_titlesong = new MP3File("snd/music/title.mp3");
  m_titlesong->open();
  LOG_MESSAGE( "done" );
  
  set_display_switch_mode(SWITCH_AMNESIA);
  set_display_switch_callback(SWITCH_IN, switchInCallback);
  set_display_switch_callback(SWITCH_OUT, switchOutCallback);
  
  LOG_MESSAGE( "Starting mp3 play..." );
  MP3Player::setMP3( m_titlesong );
  MP3Player::volume( KPlaybackVolume );
  if ( Settings::titleMusicOn == true ) {
    MP3Player::play();
  }
  LOG_MESSAGE( "done" );
  
  LOCK_FUNCTION( AudioPlayerUpdater );
  install_int_ex(AudioPlayerUpdater, BPS_TO_TIMER(35));
  
  m_quitNow = false;
  while ( !m_quitNow ) {
    int ret = m_pPage->update();
    m_pPage->redrawPage();
    
    if ( ret == MenuPage::PAGE_CHANGE ) {
      this->doPageChange();
    }
    
    int r = retrace_count + 1;
/*    do {
      if ( MP3Player::poll() == ALMP3_POLL_PLAYJUSTFINISHED ) {
        MP3Player::play();
      }
    } while ( r > retrace_count );*/
  }
  remove_int( AudioPlayerUpdater );
  remove_display_switch_callback( switchInCallback );
  remove_display_switch_callback( switchOutCallback );
}


void Game::cleanup()
{
  delete m_pPage;
}


void Game::doPageChange()
{
  int sel = m_pPage->getSelection();

  switch ( m_pPage->getPageID() ) {
    case ( MenuPage::MAINMENU ): {
      if ( sel == MenuMain::PLAY_THE_GAME ) {
        this->startNewGame();
        
      } else if ( sel == MenuMain::GOTO_LOADGAME_MENU ) {
        delete m_pPage;
        m_pPage = new MenuLoadGame();
        
      } else if ( sel == MenuMain::GOTO_SETTINGS_MENU ) {
        delete m_pPage;
        m_pPage = new MenuSettings();
        
      } else if ( sel == MenuMain::GOTO_HIGHSCORES_MENU ) {
        delete m_pPage;
        m_pPage = new HighscoresMenu();
        
      } else if ( sel == MenuMain::EXIT_THE_GAME ) {
        m_quitNow = true;
        
      } else {
        delete m_pPage;
        m_pPage = new MenuMain();
        
      }
      break;
    }
    case ( MenuPage::SETTINGS): {
      if ( sel == MenuSettings::BACK_TO_MAINMENU ) {
        delete m_pPage;
        m_pPage = new MenuMain();
        
        // Save the settings to the configuration file.
        Settings::saveGeneralSettings();
      }
      break;

    }
    case ( MenuPage::LEVELCHANGE ): {
      if ( sel == MenuLevelChange::BACK_TO_MAINMENU ) {
        // We end the current game and go back to mainmenu
        this->endCurrentGame();
        delete m_pPage;
        m_pPage = new MenuMain();
      }
      if ( sel == MenuLevelChange::PLAY_LEVEL ) {
        this->continueCurrentGame();
      }
      if ( sel == MenuLevelChange::SAVE_GAME ) {
        delete m_pPage;
        m_pPage = new MenuSaveGame();
      }
      break;
    }

    case ( MenuPage::SAVESLOT ): {
      if ( sel == MenuSaveGame::BACK_TO_LEVEL_MENU ) {
        delete m_pPage;
        MenuLevelChange* pTmp = new MenuLevelChange();
        pTmp->setLevelNum( m_nextLevel + 1 );
        m_pPage = pTmp;
      } else {
        MenuSaveGame* pTmp = dynamic_cast<MenuSaveGame*>( m_pPage );
        this->saveGame( pTmp->getSlotIndex() );
      }
      break;
    }
    case ( MenuPage::LOADGAME ): {
      if ( sel == MenuLoadGame::BACK_TO_MAINMENU ) {
        delete m_pPage;
        m_pPage = new MenuMain();
      } else {
        MenuLoadGame* pTmp = dynamic_cast<MenuLoadGame*>( m_pPage );
        this->loadGame( pTmp->getSlotIndex() );
      }
      break;
    }
    case ( MenuPage::NOTIFICATION ): {
      this->handleNotification();
      break;
    }
    case ( MenuPage::HIGHSCORES ): {
      // Go back to mainmenu
      delete m_pPage;
      m_pPage = new MenuMain();
      break;
    }
    default: {
      break;
    }
  }
}


void Game::handleNotification()
{
  MenuNotification* pTmp = dynamic_cast<MenuNotification*>( m_pPage );
  MenuPage::PageID id = pTmp->getGotoPage();
  
  switch ( id ) {
    case ( MenuPage::MAINMENU ): {
      delete m_pPage;
      m_pPage = new MenuMain();
      break;
    }
    case ( MenuPage::LEVELCHANGE ): {
      delete m_pPage;
      MenuLevelChange* pT = new MenuLevelChange();
      pT->setLevelNum( m_nextLevel + 1 );
      m_pPage = pT;
      break;
    }
    default: {
      break;
    }
  }
}


void Game::playDemo()
{
  // Stop the title-song
  remove_int( AudioPlayerUpdater );
  MP3Player::stop();
  MP3Player::setMP3( 0 );
  
  for (int l=m_nextLevel; l < 2; l++ ) {
    int reason = this->playLevel( l );
    
    for (int r=5; r < 450; r += 5) {
      circlefill(Display::buffer, 320,240, r, 0);
      Display::flip();
      int re = retrace_count + 1;
      while ( re > retrace_count );
    }
    
    if ( reason == War::QUIT_PLAYER_DIED || reason == War::QUIT_ESC ) {
      break;
    }
  }
  
  if ( exists( Consts::TMPFILE_PLAYERDATA.c_str() ) ) {
    delete_file( Consts::TMPFILE_PLAYERDATA.c_str() );
  }
  
  MP3Player::setMP3( m_titlesong );
  MP3Player::play();
  MP3Player::volume( KPlaybackVolume );
  install_int_ex(AudioPlayerUpdater, BPS_TO_TIMER(35));
}


void Game::startNewGame()
{
  // First we stop the titlesong
  remove_int( AudioPlayerUpdater );
  MP3Player::stop();
  MP3Player::setMP3( 0 );
  
  // Reset the statistics
  WarGlobals::gameStats.resetAllStats();
  WarGlobals::numberOfLives = 3;
  
  // If we start the game from the first level we show the
  // instructions.
  if ( Settings::startFromLevel == 0 ) {
    BITMAP* pB = load_bitmap( "gfx/howto.bmp", 0 );
    if ( pB != 0 ) {
      while ( true ) {
        blit( pB, screen, 0,0, 0,0, 640,480 );
        if ( key[KEY_ENTER] || key[KEY_SPACE] ) {
          break;
        }
        int r = retrace_count + 2;
        while ( retrace_count < r );
      }
      destroy_bitmap( pB );
      Sound::playSample( SMP_MENUSELECT, false );
      pB = 0;
    }
  }
  
  // Now we run the first level
  m_nextLevel = 0;
  int ret = this->playLevel( Settings::startFromLevel );
  if ( ret == War::QUIT_ESC ) {
    // User pressed ESC so we return to mainmenu
    this->endCurrentGame();

    delete m_pPage;
    m_pPage = new MenuMain();

    MP3Player::setMP3( m_titlesong );
    MP3Player::play();
    MP3Player::volume( KPlaybackVolume );
    install_int_ex(AudioPlayerUpdater, BPS_TO_TIMER(35));
    return;
    
  } else if ( ret == War::QUIT_LEVEL_COMPLETE ) {
    // Player finished the level normally. We show the level menu
    delete m_pPage;
    MenuLevelChange* tmp = new MenuLevelChange();
    tmp->setLevelNum( 2 );
    m_pPage = tmp;
    m_nextLevel = 1;
    
    MP3Player::setMP3( m_titlesong );
    MP3Player::play();
    MP3Player::volume( KPlaybackVolume );
    install_int_ex(AudioPlayerUpdater, BPS_TO_TIMER(35));
    return;
    
  } else if ( ret == War::QUIT_PLAYER_DIED ) {
    // Player died. We return to main menu
    this->checkForHighscores();
    MP3Player::setMP3( m_titlesong );
    MP3Player::play();
    MP3Player::volume( KPlaybackVolume );
    install_int_ex(AudioPlayerUpdater, BPS_TO_TIMER(35));
    return;
    
  } else if ( ret == War::QUIT_OBJECTIVE_FAILED ) {
    this->checkForHighscores();
    MP3Player::setMP3( m_titlesong );
    MP3Player::play();
    MP3Player::volume( KPlaybackVolume );
    install_int_ex(AudioPlayerUpdater, BPS_TO_TIMER(35));
    return;
    
  }
}


void Game::endCurrentGame()
{
  if ( exists( Consts::TMPFILE_PLAYERDATA.c_str() ) ) {
    delete_file( Consts::TMPFILE_PLAYERDATA.c_str() );
  }
}


void Game::continueCurrentGame()
{
  remove_int( AudioPlayerUpdater );
  MP3Player::stop();
  MP3Player::setMP3( 0 );
  
  int ret = this->playLevel( m_nextLevel );
  
  MP3Player::setMP3( m_titlesong );
  MP3Player::play();
  MP3Player::volume(160);
  install_int_ex(AudioPlayerUpdater, BPS_TO_TIMER(35));
  
  switch ( ret ) {
    case (War::QUIT_LEVEL_COMPLETE): {
      m_nextLevel += 1;
      if ( m_nextLevel >= Settings::levelList.filecount() ) {
        // All the levels has been played through
        this->endCurrentGame();
        this->checkForHighscores();
      } else {
        MenuLevelChange* pTmp = dynamic_cast<MenuLevelChange*>( m_pPage );
        pTmp->setLevelNum( m_nextLevel + 1 );
      }
      break;
    }
    case ( War::QUIT_PLAYER_DIED ): {
      // Player died. If he still has some lives left, we allow him to continue
      WarGlobals::numberOfLives -= 1;
      if ( WarGlobals::numberOfLives <= 0 ) {
        this->endCurrentGame();
        this->checkForHighscores();
      }
      break;
    }
    case ( War::QUIT_ESC ): {

      break;
    }
    default: {
      break;
    }
  }
}


void Game::saveGame( int slot )
{
  if ( slot < 0 ) {
    return;
  }
  
  SaveSlot sl( slot );
  string file = sl.getSlotFile();
  
  // Now we need to load the playerdata  
  FileHasher hashChecker( Consts::TMPFILE_PLAYERDATA.c_str() );
  if ( hashChecker.check() ) {
    // The temporary file holding the playerdata has been modified.
    alert("Failed to save game",0,0, "ok",0, 0,0);
    return;
  }
  
  Player* pPla = new Player();
  ifstream fin( Consts::TMPFILE_PLAYERDATA.c_str() );
  pPla->loadPlayerData( fin );
  fin.close();
  
  // Open the savefile
  ofstream fout( file.c_str() );
  
  if ( !fout ) {
    return;
  }
  
  fout << "[WE_WANT_WAR_SAVEFILE]" << endl;
  fout << "level: " << m_nextLevel << endl;
  fout << "lives: " << WarGlobals::numberOfLives << endl;
  fout << "difficulty: " << Settings::difficultyLevel << endl;
  
  // Save the playerdata
  pPla->savePlayerData( fout );
  
  // Save the current statistics
  WarGlobals::gameStats.writeTo( fout );
  
  // Close the file
  fout << "[END]" << endl;
  fout.close();
  
  // And add the hashcode
  FileHasher fileHasher( file );
  fileHasher.set();
  
  // Show the Game Saved-screen
  delete m_pPage;
  MenuNotification* pP = new MenuNotification("game saved", "press space" );
  pP->setGotoPage( MenuPage::LEVELCHANGE );
  m_pPage = pP;
}


void Game::loadGame( int slot )
{
  if ( slot < 0 ) {
    return;
  }
  
  SaveSlot sl( slot );
  if ( sl.getSlotContent() < 0 ) {
    return;
  }
  
  // Check the hashcode of the savefile.
  string filename = sl.getSlotFile();
  FileHasher hashChecker( filename );
  if ( hashChecker.check() ) {
    delete m_pPage;
    MenuNotification* pP = new MenuNotification("loading failed",
                                                "savefile has been corrupted" );
    pP->setGotoPage( MenuPage::MAINMENU );
    m_pPage = pP;
    return;
  }
  
  ifstream fin( filename.c_str() );
  if ( !fin ) {
    alert("Loading failed",0,0, "ok",0, 0,0);
    return;
  }
  
  string tmp;
  fin >> tmp;
  if ( tmp != "[WE_WANT_WAR_SAVEFILE]" ) {
    fin.close();
    alert("Loading failed",0,0, "ok",0, 0,0);
    return;
  }
  
  // Read the level number
  fin >> tmp >> tmp;
  m_nextLevel = atoi( tmp.c_str() );
  
  // Read number of lives
  fin >> tmp >> tmp;
  WarGlobals::numberOfLives = atoi( tmp.c_str() );
  
  // Read difficulty level
  fin >> tmp >> tmp;
  Settings::difficultyLevel = atoi( tmp.c_str() );
  
  // Now we read the playerdata.
  Player* pTmpPlayer = new Player();
  pTmpPlayer->loadPlayerData( fin );
  
  // Read the statistics
  fin >> tmp;
  if ( tmp == "<wewantwar_statistics>" ) {
    WarGlobals::gameStats.readFrom( fin );
  }
  
  // And save the playerdata to temporary file
  ofstream fout( Consts::TMPFILE_PLAYERDATA.c_str() );
  pTmpPlayer->savePlayerData( fout );
  fout << "[END]" << endl;
  
  fout.close();
  fin.close();
  
  FileHasher tmpHasher( Consts::TMPFILE_PLAYERDATA.c_str() );
  tmpHasher.set();
  
  // And now we go to levelchange menu
  delete m_pPage;
  MenuNotification* pP = new MenuNotification("game loaded", "press space" );
  pP->setGotoPage( MenuPage::LEVELCHANGE );
  m_pPage = pP;
  /*
  delete m_pPage;
  MenuLevelChange* pM = new MenuLevelChange();
  pM->setLevelNum( m_nextLevel + 1 );
  m_pPage = pM;
  */
}


int Game::playLevel(int levelnum)
{
  War* pWar = new War;
  pWar->prepareLevel( levelnum );
  pWar->play();
  
  int endCause = pWar->getEndReason();
  pWar->destroyLevel();
  
  delete pWar;
  
  return endCause;
}


void Game::checkForHighscores()
{
  // Delete the current menupage
  delete m_pPage;
  
  // Load the current highscoretable and check if we managed to get there.
  HighscoreTable htable( Consts::HIGHSCOREFILE_NAME );
  int score = WarGlobals::gameStats.score();
  
  m_pPage = new EnterHighscoresMenu( score );
}


void Game::switchInCallback()
{
  gameHasFocus = true;
  MP3Player::volume( KPlaybackVolume );
  MP3Player::play();
}

void Game::switchOutCallback()
{
  gameHasFocus = false;
  MP3Player::volume( 0 );
  MP3Player::stop();
}

void Game::AudioPlayerUpdater()
{
  if ( gameHasFocus == true ) {
    if ( MP3Player::poll() == ALMP3_POLL_PLAYJUSTFINISHED ) {
      MP3Player::play();
    }
  }
}

} // end of namespace


/**
 * Version history
 * ===============
 * $Log: game.cpp,v $
 * Revision 1.6  2006/08/16 21:12:11  lamminsa
 * Made an interrupt that manages the mp3 playback.
 *
 * Revision 1.5  2006/08/13 10:34:16  lamminsa
 * Title music volume increased.
 *
 * Revision 1.4  2006/07/27 20:30:21  lamminsa
 * MP3 playback volume adjustments.
 *
 * Revision 1.3  2006/03/29 22:29:35  lamminsa
 * Code cleanup.
 *
 * Revision 1.2  2006/02/08 21:42:57  lamminsa
 * Implemented totally new graphic and animation system.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:43  lamminsa
 * no message
 *
 * Revision 1.1  2005-12-30 23:42:17+02  lamminsa
 * No title musics if that setting is off.
 *
 * Revision 1.0  2005-11-06 01:15:39+02  lamminsa
 * Initial revision
 *
 */
